/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   copyright            : (C) 2003 by Zhang Yong                         *
 *   email                : z-yong163@163.com                              *
 ***************************************************************************/

#include "httpproxy.h"
#include "nullproxy.h"
#include "base64.h"
#include "udpsession.h"
#include "icqlog.h"

enum {
	HTTP_STATUS_NOT_CONN,
	HTTP_STATUS_ESTABLISHED
};


HTTPProxy::HTTPProxy(PROXY_INFO &info, UDPSession *session)
{
	proxy_info = info;
	udpSession = session;

	listener = NULL;
	cachedEvent = 0;
	status = HTTP_STATUS_NOT_CONN;
	bufSize = 0;

	tcpSocket = new NullProxy;
}

HTTPProxy::~HTTPProxy()
{
	delete tcpSocket;
}

void HTTPProxy::fireEvent(int event)
{
	if (!listener)
		return;
	
	event &= cachedEvent;

	if (event == READ)
		listener->onSocketRead();
	else if (event == WRITE)
		listener->onSocketWrite();
	else if (event == EXCEPTION) {
		close();
		listener->onSocketException();
	}
}

void HTTPProxy::selectEvent(int event)
{
	cachedEvent = event;

	if (status == HTTP_STATUS_ESTABLISHED)
		tcpSocket->selectEvent(event);
}

bool HTTPProxy::create(int type, SocketListener *l)
{
	listener = l;

	if (!tcpSocket->create(SOCK_STREAM, this))
		return false;

	tcpSocket->selectEvent(READ | WRITE | EXCEPTION);
	return true;
}

void HTTPProxy::connect(const char *host, uint16 port)
{
	dest_host = host;

	tcpSocket->connect(proxy_info.host.c_str(), proxy_info.port);
}

int HTTPProxy::send(const char *data, int size)
{
	uint16 n = htons(size);
	if (tcpSocket->send((const char *) &n, sizeof(n)) < 0)
		return -1;
	
	return tcpSocket->send(data, size);
}

int HTTPProxy::receive(char *data, int n)
{
	ICQ_LOG("Never call this method.\n");
	return 0;
}

void HTTPProxy::onSocketWrite()
{
	if (status == HTTP_STATUS_ESTABLISHED) {
		fireEvent(WRITE);
		return;
	}

	tcpSocket->selectEvent(READ);

	// Send 'CONNECT' request
	string req = "CONNECT ";
	req += proxy_info.host + ":443";
	req += " HTTP/1.0\r\n";
	req += "User-agent: MyICQ\r\n";

	if (!proxy_info.name.empty()) {
		// Encode using base64
		string str = proxy_info.name + ":" + proxy_info.passwd;
		char *code = NULL;
		encode_base64(str.c_str(), str.length(), &code);
		if (code) {
			req += "Proxy-authorization: Basic ";
			req += code;
			req += "\r\n";

			free(code);
		}
	}
	req += "\r\n";

	ICQ_LOG("http request:\n%s\n", req.c_str());

	tcpSocket->send(req.c_str(), req.length());
}

void HTTPProxy::onSocketRead()
{
	// Since we are already connected, receive real packet data
	if (status == HTTP_STATUS_ESTABLISHED) {
		recvPacket();
		return;
	}

	int n = tcpSocket->receive(buffer, sizeof(buffer));
	if (n <= 0) {
		fireEvent(EXCEPTION);
		return;
	}

	buffer[n] = '\0';
	ICQ_LOG("http response:\n%s\n", buffer);

	char *seps = " ";
	char *p = strtok(buffer, seps);
	if (p)
		p = strtok(NULL, seps);
	if (!p || *p != '2') {
		fireEvent(EXCEPTION);
		return;
	}

	ICQ_LOG("http tunnel has been established.\n");

	status = HTTP_STATUS_ESTABLISHED;

	fireEvent(WRITE);
	tcpSocket->selectEvent(cachedEvent);
}

void HTTPProxy::onSocketException()
{
	fireEvent(EXCEPTION);
}

/* Since TCP connection is really a data stream and there is no packet boundary,
 * extra things must be done. we put an extra 2 bytes(WORD) at the head of each
 * packet describing its size.
 */
void HTTPProxy::recvPacket()
{
	int n;

	while ((n = tcpSocket->receive(buffer + bufSize, sizeof(buffer) - bufSize)) > 0) {
		bufSize += n;
		char *start = buffer;
		char *end = start + bufSize;
		uint16 len;

		while (start + sizeof(len) < end) {
			len = ntohs(*(uint16 *) start);
			if (end - start - sizeof(len) < len)
				break;

			start += sizeof(len);

			if (len >= sizeof(UDP_PACKET_HDR)) {
				UDPInPacket in(start, len);
				udpSession->onPacketReceived(in);
			}

			start += len;
		}

		bufSize = end - start;
		if (bufSize > 0)
			memcpy(buffer, start, bufSize);
	}

	if (n == 0)
		fireEvent(EXCEPTION);
}
