/*
 * GTK See -- an image viewer based on GTK+
 * Copyright (C) 1998 Hotaru Lee <jkhotaru@mail.sti.com.cn> <hotaru@163.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* Code based in Gimp Specification for gbr and pat files
 *
 * Support only gbr version 2
 *
 * 2004-11-02: First development
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>

#include "im_gmp.h"

#define PAT_MAGIC (('G' << 24) | ('P' << 16) | ('A' << 8) | ('T' << 0))
#define GBR_MAGIC (('G' << 24) | ('I' << 16) | ('M' << 8) | ('P' << 0))

#define ToL(buffer, off)   (buffer[(off)+3] | buffer[(off)+2]<<8 | buffer[(off)+1]<<16 | buffer[(off)+0]<<24)

gboolean pat_load (pat_info *, FILE *, GmpLoadFunc );
gboolean gbr_load (gbr_info *, FILE *, GmpLoadFunc );



/* Begin the code... */
gboolean
pat_get_header(gchar *filename, pat_info *info)
{
   FILE     *fd;
   guchar   buf[24];
   gboolean status = FALSE;

   if ((fd = fopen (filename, "rb")) == NULL)
   {
      return FALSE;
   }

   if (fread (buf, sizeof (pat_info), 1, fd) == 1)
   {
      info->headersize = ToL(buf, 0 );
      info->version    = ToL(buf, 4 );
      info->width      = ToL(buf, 8 );
      info->height     = ToL(buf, 12);
      info->depth      = ToL(buf, 16);
      info->magicnumber= (buf[20] << 24) + (buf[21] << 16) +
                           (buf[22] << 8) + (buf[23] << 0) ;

      if (info->version == 1                  &&
            info->width > 0                   &&
            info->height> 0                   &&
            (info->depth>=1 && info->depth<=4)&&
            info->magicnumber == PAT_MAGIC)
      {
         status = TRUE;
      }
   }

   fclose(fd);
   return status;
}

gboolean
gbr_get_header(gchar *filename, gbr_info *info)
{
   FILE     *fd;
   guchar   buf[24];
   gboolean status = FALSE;

   if ((fd = fopen (filename, "rb")) == NULL)
   {
      return FALSE;
   }

   if (fread (buf, sizeof (gbr_info), 1, fd) == 1)
   {
      info->headersize = ToL(buf, 0 );
      info->version    = ToL(buf, 4 );
      info->width      = ToL(buf, 8 );
      info->height     = ToL(buf, 12);
      info->depth      = ToL(buf, 16);
      info->magicnumber= (buf[20] << 24) + (buf[21] << 16) +
                           (buf[22] << 8) + (buf[23] << 0) ;

      if (info->version == 2                  &&
            info->width > 0                   &&
            info->height> 0                   &&
            (info->depth==1 || info->depth==4)&&
            info->magicnumber == GBR_MAGIC)
      {
         status = TRUE;
      }
   }

   fclose(fd);
   return status;
}

gboolean
pat_load (pat_info *pathdr, FILE *fd, GmpLoadFunc func)
{
   gint     *dest, transparent;
   guchar   *tmp, buf[4], *nop[0];
   gint     i, j, k;

   /* Move the pointer to the raw data */
   fseek(fd, (glong) pathdr->headersize, SEEK_SET);

   transparent = 256;
   nop[0]      = NULL;
   dest = g_malloc(sizeof(gint)   * pathdr->width * 4);
   tmp  = g_malloc(sizeof(guchar) * pathdr->width * 4);

   switch (pathdr->depth)
   {
      case 1: /* PAT raw greyscale */
         for (i = 0; i < pathdr->height; i++)
         {
            if (!fread(tmp, pathdr->width, sizeof(guchar), fd)) break;
            for (j=0, k=0; j < pathdr->width * 3; j+=3, k++)
            {
               dest[j+0] = tmp[k];
               dest[j+1] = tmp[k];
               dest[j+2] = tmp[k];
            }
            if ((*func) (dest, nop, transparent, pathdr->width, i, 0)) break;
         }
         break;

      case 2: /* PAT raw greyscale with channel A */
         for (i = 0; i < pathdr->height; i++)
         {
            for (j=0; j < pathdr->width * 3; j+=3)
            {
               if (!fread(&buf, 2, sizeof(guchar), fd)) break;
               if (buf[1] == 0xFF)
               {
                  dest[j+0] = buf[0];
                  dest[j+1] = buf[0];
                  dest[j+2] = buf[0];
                  
               } else
               {
                  dest[j+0] = transparent;
                  dest[j+1] = transparent;
                  dest[j+2] = transparent;
               }
            }
            if ((*func) (dest, nop, transparent, pathdr->width, i, 0)) break;
         }
         break;

      case 3: /* PAT raw rgb */
         for (i = 0; i< pathdr->height; i++)
         {
            if (!fread(tmp, pathdr->width * 3, 1, fd)) break;
            for (j=0; j < pathdr->width * 3; j++)
            {
               dest[j] = tmp[j];
            }
            if ((*func) (dest, nop, transparent, pathdr->width, i, 0)) break;
         }
         break;

      case 4: /* PAT raw rgb with Alpha channel */
         for (i = 0; i< pathdr->height; i++)
         {
            for (j=0; j < pathdr->width * 3; j+=3)
            {
               if (!fread(&buf, 4, sizeof(guchar), fd)) break;
               if (buf[3] == 0xFF)
               {
                  dest[j+0] = buf[0];
                  dest[j+1] = buf[1];
                  dest[j+2] = buf[2];
               } else
               {
                  dest[j+0] = transparent;
                  dest[j+1] = transparent;
                  dest[j+2] = transparent;
               }
            }
            if ((*func) (dest, nop, transparent, pathdr->width, i, 0)) break;
         }
         break;
   }

   g_free(dest);
   g_free(tmp);
   return TRUE;
}

gboolean
gbr_load (gbr_info *gbrhdr, FILE *fd, GmpLoadFunc func)
{
   guint    *dest, transparent;
   guchar   *tmp, buf[4], *nop[0];
   gint     i, j, k;

   /* Move the pointer to the raw data */
   fseek(fd, (glong) gbrhdr->headersize, SEEK_SET);

   transparent = 256;
   nop[0]      = NULL;
   dest = g_malloc(sizeof(guint)  * gbrhdr->width * 4);
   tmp  = g_malloc(sizeof(guchar) * gbrhdr->width * 4);

   switch (gbrhdr->depth)
   {
      case 1: /* GBR raw greyscale */
         for (i = 0; i < gbrhdr->height; i++)
         {
            if (!fread(tmp, gbrhdr->width, sizeof(guchar), fd)) break;
            for (j=0, k=0; j < gbrhdr->width * 3; j+=3, k++)
            {
               dest[j+0] = tmp[k];
               dest[j+1] = tmp[k];
               dest[j+2] = tmp[k];
            }
            if ((*func) (dest, nop, transparent, gbrhdr->width, i, 0)) break;
         }
         break;

      case 4: /* GBR raw rgb with Alpha channel */
         for (i = 0; i< gbrhdr->height; i++)
         {
            for (j=0; j < gbrhdr->width * 3; j+=3)
            {
               if (!fread(&buf, sizeof(guchar), 4, fd)) break;
               if (buf[3] == 0xFF)
               {
                  dest[j+0] = buf[0];
                  dest[j+1] = buf[1];
                  dest[j+2] = buf[2];
               } else
               {
                  dest[j+0] = transparent;
                  dest[j+1] = transparent;
                  dest[j+2] = transparent;
               }
            }
            if ((*func) (dest, nop, transparent, gbrhdr->width, i, 0)) break;
         }
         break;

   }

   g_free(dest);
   g_free(tmp);
   return TRUE;
}

gboolean
gmp_load (gchar *filename, GmpLoadFunc func)
{
   FILE     *fd;
   pat_info pathdr;
   gbr_info gbrhdr;

   if (pat_get_header(filename, &pathdr) == TRUE)
   {
      fd = fopen (filename, "rb");
      return pat_load (&pathdr, fd, func);
   } else
   {
      if (gbr_get_header(filename, &gbrhdr) == TRUE)
      {
         fd = fopen (filename, "rb");
         return gbr_load (&gbrhdr, fd, func);
      } else
      {
         return FALSE;
      }
   }

   return TRUE;
}

