/*
 * GTK See -- an image viewer based on GTK+
 * Copyright (C) 1998 Hotaru Lee <jkhotaru@mail.sti.com.cn> <hotaru@163.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * 2003-09-14: Modifications in:
 *             remove_file function
 *             remove_it   function
 *             rename_it   function
 * 2004-05-13: Added code for rename multiple files
 */
#include "config.h"

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <gtk/gtk.h>

#include "intl.h"
#include "common_tools.h"
#include "dialog.h"
#include "imagelist.h"
#include "gtksee.h"
#include "rc.h"
#include "rename_seq.h"

void     close_dialog   (GtkWidget *widget);

GdkColor
gint_to_gdkcolor(gint intcolor)
{
   GdkColor color;

   color.pixel = 0L;
   color.red   = (gint) (((gdouble)((intcolor & 0xff0000) >> 16) / 255.0) * 65535.0);
   color.green = (gint) (((gdouble)((intcolor & 0x00ff00) >>  8) / 255.0) * 65535.0);
   color.blue  = (gint) (((gdouble)((intcolor & 0x0000ff)      ) / 255.0) * 65535.0);

   return color;
}

void
set_widget_bg_color(GtkWidget *widget, gint newcolor)
{
   GtkStyle *style;
   GdkColor color;

   style = gtk_style_copy(gtk_widget_get_default_style());
   color = gint_to_gdkcolor(newcolor);

   style->bg[GTK_STATE_NORMAL] = color;
   gtk_widget_set_style(widget, style);

}

gchar *
stripslash(gchar *s)
{
   gint  i, j, len;

   len = strlen(s);
   for (i=1, j=1; i<=len; i++, j++)
   {
      if (s[i] == '/' && s[i-1] == s[i])
      {
         j --;
      }
      s[j] = s[i];
   }

   if (s[j-2] == '/')
   {
      s[j-2] = '\0';
   }

   return (gchar *) s;
}

void
close_gtksee()
{
   rc_save_gtkseerc();
   gtk_main_quit();
}

void
error_dialog(GtkWidget *widget, gchar *myline, gchar *usr_msg)
{
   dialog_message(widget, _("Error!"),
                  myline, usr_msg,
                  OK_DIALOG,
                  WARNING_DIALOG,
                  FALSE, NULL, 0);

}

void
alert_dialog(GtkWidget *widget, gchar *myline, gchar *usr_msg)
{
   dialog_message(widget, _("Warning"),
                  myline, usr_msg,
                  OK_DIALOG,
                  WARNING_DIALOG,
                  FALSE, NULL, 0);

}

void
close_dialog(GtkWidget *widget)
{
   gtk_grab_remove(widget);
   gtk_widget_destroy(widget);
}

void
remove_file(GtkWidget *il, GList *selection)
{
   gchar *buffer;
   gint  result, total;

   buffer = gtksee_malloc(BUF_MIN);
   total  = g_list_length(selection);
   sprintf(buffer, _("Do you really want to remove %i file(s) ?"), total);

   result = dialog_message(il, _("Remove file(s)"),
                           buffer, NULL,
                           OK_CANCEL_DIALOG,
                           DELETE_DIALOG,
                           FALSE, NULL, 0);
   g_free(buffer);

   if (result == OK_STATUS)
   {
      ImageInfo   *info;
      GList       *itemselec;
      guint       focus_row, row_list, row = 0;
      gint        filesdel;

#ifdef GTK_HAVE_FEATURES_1_1_0
      focus_row = GTK_CLIST(IMAGE_LIST(il)->child_list)->focus_row;
#endif
      itemselec = selection;
      filesdel  = 0;

      /* Delete the files */
      buffer = gtksee_malloc(BUF_STD);

      while (selection != NULL)
      {
         info = image_list_get_by_serial(IMAGE_LIST(il),
                                          (guint)(selection)->data);

         sprintf(buffer, "%s/%s", IMAGE_LIST(il)->dir, info->name);

         if (unlink(buffer) != -1)
         {
            filesdel ++;
            if (info->serial < focus_row)
               row ++;
         } else
         {
            if (total == 1)
            {
               error_dialog(il, _("Problem deleting file!"), strerror(errno));
            } else
            {
               sprintf(buffer, "%s %s", _("Problem deleting"), info->name);
               result = dialog_message(il, _("Warning"),
                                 buffer,
                                 _("Do you want continue?"),
                                 OK_CANCEL_DIALOG,
                                 QUESTION_DIALOG,
                                 FALSE,
                                 NULL, 0);
               if(result == CANCEL_STATUS)
               {
                  goto stop_delete;
               }
            }
         }

         selection = g_list_next(selection);
      }

stop_delete:
      g_free(buffer);
      if (filesdel != 0)
      {
         refresh_list();
         row_list = GTK_CLIST(IMAGE_LIST(il)->child_list)->rows;

#ifdef GTK_HAVE_FEATURES_1_1_0
         gtk_clist_unselect_row(GTK_CLIST(IMAGE_LIST(il)->child_list),
                                 focus_row, 0);
         focus_row = focus_row - row;

         if (row_list >= 0)
         {
            if (focus_row >= row_list - 1)
               focus_row = row_list - 1;
            GTK_CLIST(IMAGE_LIST(il)->child_list)->focus_row = focus_row;
            gtk_clist_select_row(GTK_CLIST(IMAGE_LIST(il)->child_list),
                                 focus_row, 0);
            gtk_clist_moveto(GTK_CLIST(IMAGE_LIST(il)->child_list),
                                 focus_row, 0, 0, 0);
         }
#else
         if (row_list >= 0)
         {
            gtk_clist_select_row(GTK_CLIST(IMAGE_LIST(il)->child_list), 0, 0);
            gtk_clist_moveto(GTK_CLIST(IMAGE_LIST(il)->child_list),
                              focus_row, 0, 0, 0);
         }
#endif
      }
   }

}

void
rename_file(GtkWidget *il, GList *selection)
{
   if (g_list_length(selection) == 1)
   {
      ImageInfo *info;
      gchar     *str;
      gint      result;

      info   = image_list_get_by_serial(IMAGE_LIST(il),
                                        (guint)(selection)->data);
      result = dialog_message(il, _("Rename"),
                              _("Rename file:"), info->name,
                              OK_CANCEL_DIALOG,
                              RENAME_DIALOG,
                              TRUE,
                              &str, 10);

      if (result == OK_STATUS)
      {
         gchar       *fromfile, *tofile;
         struct stat stbuf;

         fromfile = gtksee_malloc(BUF_STD);
         tofile   = gtksee_malloc(BUF_STD);

         if (strncmp(str, info->name, 256) != 0)
         {
            sprintf(fromfile, "%s/%s", IMAGE_LIST(il)->dir, info->name);
            sprintf(tofile,   "%s/%s", IMAGE_LIST(il)->dir, str);

            if (stat(tofile, &stbuf) != -1) /* Exist tofile ? */
            {
               error_dialog(il, _("Problem renaming file!"),
                            _("File exist !"));   /* Yes */
            } else
            {
               if (rename(fromfile, tofile) == 0)
               {
                  strncpy(info->name, str, 256);
                  refresh_list();
               } else
               {
                  error_dialog(il, _("Problem renaming file!"),
                               strerror(errno));
               }

            }
         } else
         {
            error_dialog(il, _("Problem renaming file!"),
                         _("Same filename !"));
         }

         g_free(fromfile);
         g_free(tofile);
         g_free(str);
      }
   } else /* Rename a file selection */
   {
      rename_serie(il, selection);
   }

}

gulong
buf2long(guchar *buf, gint n)
{
   gulong l;
   gulong byte_order = 1;

   if ((char) byte_order)
   {
      l  = (gulong) (buf[n+0] << 24);
      l |= (gulong) (buf[n+1] << 16);
      l |= (gulong) (buf[n+2] <<  8);
      l |= (gulong)  buf[n+3];
   } else
   {
      l  = (gulong)  buf[n+0] ;
      l |= (gulong) (buf[n+1] <<  8);
      l |= (gulong) (buf[n+2] << 16);
      l |= (gulong) (buf[n+3] << 24);
   }

   return (l);
}

gushort
buf2short(guchar *buf, gint n)
{
   gushort l;
   gulong  byte_order = 1;

   if ((char) byte_order)
   {
      l  = (gushort) (buf[n+0] << 8);
      l |= (gushort)  buf[n+1];
   } else
   {
      l  = (gushort)  buf[n+0];
      l |= (gushort) (buf[n+1] << 8);
   }

   return (l);
}
